% main_estimation_mcmc.m
% 
% main MCMC estimation file 
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %

% -- Clear Workspace -- %
clc;
clear;
close all;
rng(61824)

% -- Options -- %
addpath('../Solver - SV');      % add path with solution code 
addpath(genpath('../Utilities'));
GLOBALS = globals();    % load structure of GLOBALS

% -- Filtering Type -- %
kf_filtering_level = 0;

% -- MCMC Draw Options -- %
burn        = 1;
ndraw       = 10;

% -- Set System Dimensions -- %
ny = GLOBALS.ny;   % # of control variables
nx = GLOBALS.nx;   % # of state variables
ne = GLOBALS.ne;   % # of structural shocks
ns = GLOBALS.ns;   % # of regimes

% -- Set Estimation Dimensions -- %
NX = GLOBALS.estimation.NX;   % # of states in estimation

% - Data -- %
y       = GLOBALS.y;
dates   = GLOBALS.dates;

%% -- MCMC INITIALIZATION -- %%
% -- Set MCMC Arrays -- %
THETA           = getparams(GLOBALS);
nparm           = length(THETA);            % number of parameters to estimate
draw            = NaN(nparm,ndraw+burn);    % matrix of MCMC draws
cdraw           = NaN(nparm,ndraw+burn);
logposteriors   = NaN(1,ndraw+burn);


% -- Compute Hessian -- %
% hessian = num_hess('endo_ms_lik_2nd_kim_nelson_SV',THETA,0,y,1,1,0,GLOBALS);
% SIGMA = inv(hessian);
% V2 = chol(project(SIGMA,0))';

V2 = 0.0001*diag(THETA);


%% -- RUN MCMC -- %%

% -- Run Filter on Mode of Parameters -- %
mode_value = endo_ms_lik_2nd_kim_nelson_SV(THETA,y,kf_filtering_level,1,0,GLOBALS);

% -- Initialize MCMC Objects -- %
naccept     = 0;                    % number of acceptances
iterratio   = 0;                    % number of iterations with valid draws
ratio       = 0;                    % ratio of acceptences
aprob       = 0;                    % acceptance prob.
optimal     = 0.25;                 % target acceptance ratio for MCMC


% -- MCMC Iteration -- %
tic;

% -- First Draw -- %s
draw(:,1)   = THETA + V2*randn(nparm,1);
newpost = endo_ms_lik_2nd_kim_nelson_SV(draw(:,1),y,kf_filtering_level,1,0,GLOBALS);
logposteriors(1) = newpost;


for iter = 2:ndraw+burn
    disp(iter);

    itertic = tic;

    % -- Overwrite Old Likelihood and Prior -- %
    oldpost = newpost;    % overwrite posterior

    % -- Draw a New Value -- %
    cdraw(:,iter)   = draw(:,iter-1) + V2*randn(nparm,1);

    % -- Check the Draw's Prior -- %
    [prior,rflg] = priorval(cdraw(:,iter),GLOBALS);  % rflag=1 if 0 probability draw

    % -- Run Filter on New Draw -- %
    if rflg == 0
        [proppost, ~, ~, retcode] = ...
            endo_ms_lik_2nd_kim_nelson_SV(cdraw(:,iter),y,kf_filtering_level,1,0,GLOBALS);
    else
        retcode = 0;
    end

    % -- Discard if No  Solution, Otherwise Proceed -- %
    if retcode == 0
        draw(:,iter)    = draw(:,iter-1);
        proppost        = -Inf;
        aprob           = 0;
    else

        % -- Construct Posterior Ratio -- %
        aprob       = min(1,exp(proppost-oldpost));

        % -- Draw a Random Number, Accept if Ratio Exceeds Random -- %
        if rand(1,1) <= aprob

            % -- Saving Draw Features -- %
            draw(:,iter)    = cdraw(:,iter);    % save draw
            naccept         = naccept+1;        % increment number of acceptances
            newpost         = proppost;  

        % -- Reject Otherwise -- %
        else
            draw(:,iter)    = draw(:,iter-1);
            newpost         = oldpost;
        end
    end

        % -- Update Scale -- %
        gamma   = (iter-1)^(-51/100);
        s       = gamma*(aprob-optimal);

        % -- Update Proposal Density Covariance Matrix -- %
        delta_X     = randn(nparm,1);
        V           = V2*(eye(nparm)+s*(delta_X*delta_X'/(delta_X'*delta_X)))*V2';
        L           = tril(V,-1); 
        V           = diag(diag(V))+L+L'; % This help keeping V symmetrical
        V2  	    = chol(V)';

    % -- Save Posteriors -- %
    logposteriors(iter) = newpost;

    % -- Track Acceptance Ratio -- %
    iterratio   = iterratio + 1;
    ratio       = naccept/iterratio;

    itertoc=toc(itertic);

end
    


%% -- Results -- %%


% -- Save Draws -- %
fulldraw        = draw;
draw            = draw(:,burn+1:burn+ndraw);
logposteriors   = logposteriors(burn+1:burn+ndraw);

% -- Thinning -- %
draw = draw(:,1:1:end);     % 100

% -- Percentiles -- %
q05         = prctile(draw,5,2);
q50         = prctile(draw,50,2);
q95         = prctile(draw,95,2);


% -- Print to TEX -- %
disp(' ')
delete ../Figures and Tables/Table_2.tex;
diary('../Figures and Tables/Table_2.tex');
for ii = 1:length(THETA)
    latexPrt([GLOBALS.parmlist{ii,1} ' & ' GLOBALS.parmlist{ii,2} ' & ' GLOBALS.parmlist{ii,3} , '(' num2str(GLOBALS.parmlist{ii,4}) ',' num2str(GLOBALS.parmlist{ii,5}) ')'], ...
       [THETA(ii) q05(ii) q50(ii) q95(ii)],4)
end
diary off;
